/*******************************************************************************
********************************************************************************
** COPYRIGHT:    (c) 2013 Rohde & Schwarz
** MODULE:       FileAccessExample.cpp
** LANGUAGE:     C/C++
** AUTHOR:       
** ABSTRACT:     Example for PR100, on how to access the storage card via SCPI
** PREMISES:
** REMARKS:
** HISTORY:
********************************************************************************/

/* INCLUDE FILES ***************************************************************/
#include "CSCPIHelper_PR100.h"
#include "CSCPIFileAccess_PR100.h"
#include <string>
#include <map>
#include <vector>
#include <sstream>

#ifndef _WINSOCKAPI_
#include <winsock2.h>
#endif

using namespace std;

#define STATE_IDLE  (0)
#define STATE_BUSY  (1)
#define STATE_DONE  (2)
#define STATE_ERROR (3)

//Structure used by callback
typedef struct
{
    FILE * filedesc;    //FILE descriptor. If NULL => no transfer in progress.
    int    nFileIndex;  //The nth file to be transferred
    int    nLenWritten; //
    int    nState;      //One of the STATE_* variables

} CALLBACK_DATA_T;

// Callback function for getting block data
void download_callback(void* pPrivate, const CSCPIHelper::BLOCKDATA_T& data)
{
    CALLBACK_DATA_T * p_local = (CALLBACK_DATA_T * ) pPrivate;

	//sanity check
	if(pPrivate == NULL)
	{
		printf("ERROR: NULL pointer\n");
		return;
	}

    switch(data.status)
    {
    case CSCPIHelper::BLOCKDATA_STATUS_DONE:
        printf("FILE#%02d: CALLBACK: success, %d bytes written\n", p_local->nFileIndex, p_local->nLenWritten);
        p_local->nState = STATE_DONE;
        break;

    case CSCPIHelper::BLOCKDATA_STATUS_FAIL:
        printf("FILE#%02d: CALLBACK: failed, %d bytes written\n", p_local->nFileIndex, p_local->nLenWritten);
        p_local->nState = STATE_ERROR;
        break;

    case CSCPIHelper::BLOCKDATA_STATUS_DATA:
        if(p_local->filedesc)
        {
            int nLen = fwrite((void *) data.pBufferData, 1, data.bufferSize, p_local->filedesc);
            p_local->nLenWritten += nLen;
            if(nLen != data.bufferSize)
            {
                printf("FILE#%d: CALLBACK: File write error!\n", p_local->nFileIndex);
                p_local->nState = STATE_ERROR;
            }
        }
        break;

    default:
        break;
    }

}

void downloadFiles(CSCPIHelper& devSCPI, bool bDebug)
{
    CALLBACK_DATA_T callback_data;
    memset(&callback_data, 0, sizeof(CALLBACK_DATA_T));

    // Setup file access
    CSCPIFileAccess fileaccess(devSCPI, download_callback, &callback_data);
    fileaccess.SetDebugLevel(bDebug);

    std::vector<CSCPIFileAccess::FILE_PROPERTY> files;   //vector containing file properties
    std::map <std::string, std::string> filesToDownload; //map containing files to download

    //Get the list of files from "\Storage Card\<devicename>"
    std::string path = "\\Storage Card\\";
    path.append(devSCPI.DeviceName());

    if( fileaccess.listfiles(path, files) == 0)
    {
        //In this example, scan for files which match the following patterns,
        //and download them to the local directory:
        //- RecTrace_xxx.rtr
        //- RecIQ_xxx.riq
        //- RecAudio_xxx.wav
        //- Screen_xxx.png

        //Search vector for matching files to download
        for(std::vector<CSCPIFileAccess::FILE_PROPERTY>::iterator it = files.begin(); it != files.end(); ++it) 
        {
            //For debugging to print out the file details
            printf("LIST: [%s][%s][%s][%d][%s][%s]\n", it->path.c_str(), it->name.c_str(), it->ext.c_str(), it->size, it->date.c_str(), it->time.c_str());

            if (  ((it->name.find("RecTrace_") == 0) && (it->ext == "rtr"))
                || ((it->name.find("RecIQ_")    == 0) && (it->ext == "riq"))
                || ((it->name.find("RecAudio_") == 0) && (it->ext == "wav"))
                || ((it->name.find("Screen_")   == 0) && (it->ext == "png"))
                )
            {
                //Found matching pattern, append to the list of files to download
                std::stringstream ss_destname;
                std::string srcpathname = it->path + "\\" + it->name;
                ss_destname << it->devName << "_" << it->devSN << "_" << it->name;
                if(it->ext != "") 
                {
                    srcpathname += ("." + it->ext);
                    ss_destname << "." << it->ext;
                }

                //Add file to map
                filesToDownload[srcpathname] = ss_destname.str();
            }
        }

        //Download the files
        if(!filesToDownload.empty())
        {
            for(std::map <std::string, std::string>::iterator it = filesToDownload.begin(); it != filesToDownload.end(); ++it) 
            {
                const std::string& src  = it->first;
                const std::string& dest = it->second;

                //Open the destination file and set the callback data
                callback_data.filedesc      = fopen(dest.c_str(), "w+b");
                callback_data.nLenWritten   = 0;
                callback_data.nState        = STATE_IDLE;
                callback_data.nFileIndex++;

                if( callback_data.filedesc != NULL)
                {
                    printf("FILE#%02d: Downloading file [%s] -> [%s] ...\n", callback_data.nFileIndex, src.c_str(), dest.c_str());

                    //Perform the file download
                    if( fileaccess.download(src) == 0 )
                    {
                        printf("FILE#%02d: Download OK\n", callback_data.nFileIndex, src.c_str());
                    }
                    else
                    {
                        printf("FILE#%02d: Error downloading file %s from device\n", callback_data.nFileIndex, src.c_str());
                    }

                    fclose(callback_data.filedesc);
                    callback_data.filedesc = NULL;
                }
                else
                {
                    printf("FILE#%d: Unable to open destination file %s for writing\n", callback_data.nFileIndex, dest.c_str());
                }
            }
        }
        else
        {
            printf("No matching files to download\n");
        }
    }
    else
    {
        printf("No files found or SDCARD not available\n");
    }
}

//main function 
int main(int argc, char **argv)
{
    printf("PR100/EM100/DDF007 FILE ACCESS EXAMPLE 1.01, Copyright (c) 2013 Rohde&Schwarz\n\n");

	unsigned short port    = 0;
	std::string remotename = "";
    bool m_debug           = false;

	// Process command line
	bool bCmdLineOK = true;
	for(int param = 1; (param < argc) && bCmdLineOK; ++param)
	{
		if (strcmp(argv[param], "-p") == 0)
		{
			// Evaluate TCP port parameter
			if (++param < argc)
			{
				port = (unsigned short) atoi(argv[param]);
				if ((port == 0) || (port > USHRT_MAX))
				{
					bCmdLineOK = false;
				}
			}
			else
			{
				bCmdLineOK = false;
			}
		}
		else if (strcmp(argv[param], "-c") == 0)
		{
			// Evaluate SCPI device 
			if (++param < argc)
			{
				remotename = argv[param];
			}
			else
			{
				bCmdLineOK = false;
			}
		}
        else if (strcmp(argv[param], "-d") == 0)
        {
            m_debug = true;
        }
		else
		{
			bCmdLineOK = false;
		}
	}

	if(	(bCmdLineOK == false) || (port == 0) || (remotename == "") )
	{
		printf("Usage: %s -c <ipaddress> -p <port>\n", argv[0]);
        return -1;
	}

	// Check for WinSock
    WSADATA wsaData;
    if (WSAStartup(MAKEWORD(2, 0), &wsaData) != 0)
    {
        /* Tell the user that we couldn't find a usable */
        /* WinSock DLL.                  */
        printf("ERROR: Unable to find usable windows socket dll.\n");
        return -2;
    }

    /* Confirm that the WinSock DLL supports 2.0.
       Note that if the DLL supports versions greater
       than 2.0 in addition to 2.0, it will still return
       2.0 in wVersion since that is the version we
       requested.
	*/
    if ( LOBYTE(wsaData.wVersion) != 2 || HIBYTE(wsaData.wVersion) != 0 )
    {
        /* Tell the user that we couldn't find a usable */
        /* WinSock DLL.                  */
        printf("ERROR: Unable to get correct winsock version 2.0.\n");
        WSACleanup();
        return -2;
    }


    // Setup connection to SCPI device
    CSCPIHelper devSCPI(remotename, port);
    devSCPI.SetDebugLevel(m_debug);

    printf("Attempting to connect to %s@%d\n", remotename.c_str(), port);

    if(devSCPI.Connect())
    {

        // Successfully connected to SCPI device
        printf("Connected to %s %s@%d SN=%d FW=%s\n", devSCPI.DeviceName().c_str(), devSCPI.DeviceAddress().c_str(), devSCPI.DevicePort(), devSCPI.DeviceSN(), devSCPI.DeviceFWVer().c_str());
        
        // Check that RC option is supported
        unsigned int options;
        if(devSCPI.GetDeviceOptions(options))
        {
            if( (options & CSCPIHelper::OPTION_RC) == 0)
            {
                printf("RC option is not installed. It is required for file access SCPI command\n");
            }
            else
            {
                downloadFiles(devSCPI, m_debug);
            }
        }
        else
        {
            printf("Unable to get the options from the device\n");
        }

        //Disconnect from remote device
        devSCPI.Disconnect();
    }
    else
    {
        printf("ERROR: Unable to connect to %s@%d\n", remotename.c_str(), port);
    }

    WSACleanup();
    return 0;
}